/**************************************************************************
 *                                                                        *
 * libindexer 1.1                                                         *
 * Copyright (C) 2012 Fabien Menemenlis (nihilist@dead-inside.org)        *
 *                                                                        *
 * This library is free software; you can redistribute it and/or          *
 * modify it under the terms of the GNU Lesser General Public             *
 * License as published by the Free Software Foundation; either           *
 * version 2.1 of the License, or (at your option) any later version.     *
 *                                                                        *
 * This library is distributed in the hope that it will be useful,        *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 * Lesser General Public License for more details.                        *
 *                                                                        *
 * You should have received a copy of the GNU Lesser General Public       *
 * License along with this library; if not, write to the Free Software    *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  *
 * MA  02111-1307  USA                                                    *
 *                                                                        *
 **************************************************************************/

// #define NO_MULTI

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <machine/param.h>
#include <sys/types.h>
#ifndef NO_MULTI
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#endif
#include <limits.h>
#include <dirent.h>
#include <fcntl.h>

#include "idxdefines.h"
#include "libindexer.h"
#include "latintable.h"
#include "hash.h"
#include "dhash.h"

/*
 * index count before calling savetree() (to flush memory)
 */

#define MAXCOUNT 12000000 /* around 1.5 GB */

#define MAXIDX MAXCOUNT * 8

/*
 * defines for files merging
 */
#define MAXFILES 1024

#define FREE(a) if (a) { free(a); a = NULL; }


static char endchar[256] = {0}, lnkchar[256] = {0};

static s_hash *hash;

static FILE *fid; /* file containing "doc id" to "offset in data file" table */
#ifndef NO_MULTI
static void *shmptr; /* shared memory pointer */
static off_t *noff;
static int *curdocid;
static int semid; /* semaphore ID for process synchronization */
static int shmid;
#else
static off_t noff;
static int curdocid;
#endif
static int idxcount;
static int maxcount = MAXCOUNT;
static int maxidx = MAXIDX;


static void db_append_dboff(FILE *fdboff, unsigned char count, off_t off) {
    
    fwrite(&count, sizeof(unsigned char), 1, fdboff);
    fwrite(&off, 5, 1, fdboff); /* not big endian compatable, 5 is for 40 bits from the offset */
}


static void db_append_dbdata(FILE *fdbdata, unsigned short keysize, char *key, char *data) {
    
    fwrite(&keysize, sizeof(unsigned short), 1, fdbdata);
    fwrite(key, 1, keysize, fdbdata);
    fwrite(data, 1, 5 + sizeof(int) + sizeof(int), fdbdata);
}

static int indexer_init(char *pname, char *outname) {
    char *ptr;
#ifndef NO_MULTI
    key_t key;
    union semun arg;
#endif
    char filename[100 + 1];


    idxcount = 0;
    lnkchar[','] = endchar[','] = 1;
    lnkchar['\''] = endchar['\''] = 1;
    lnkchar['.'] = endchar['.'] = 1;
    lnkchar[':'] = endchar[':'] = 1;
    lnkchar[';'] = endchar[';'] = 1;
    lnkchar['#'] = 1;
    lnkchar['%'] = 1;
    lnkchar['&'] = 1;
    lnkchar['*'] = 1;
    lnkchar['$'] = 1;
    lnkchar['-'] = 1;
    lnkchar['/'] = endchar['/'] = 1;
    lnkchar['<'] = 1;
    lnkchar['>'] = 1;
    lnkchar['@'] = 1;
    lnkchar['_'] = 1;
    lnkchar['='] = 1;
    lnkchar['+'] = 1;
    
    snprintf(filename, 100, "%s.off", outname);
    if ((fid = fopen(filename, "w+")) == NULL) {
        perror(filename);
        return(1);
    }
#ifndef NO_MULTI
    key = ftok(pname, 1);

    /* free any previous shared memory or semaphore attached to this key */
    if (shmid = shmget(key, sizeof(off_t) + sizeof(int), 0)) {
        shmctl(shmid, IPC_RMID, NULL);
    }
    if (semid = semget(key, 1, 0)) {
        semctl(semid, 0, IPC_RMID);
    }

    /* semaphore initialization */
    if ((semid = semget(key, 1, 0644 | IPC_CREAT)) == -1) {
        perror("semget");
        return(1);
    }
    arg.val = 1;
    semctl(semid, 0, SETVAL, arg);

    /* create shared memory segments for next offset (off_t) and next doc id (int) */
    shmid = shmget(key, sizeof(off_t) + sizeof(int), 0644 | IPC_CREAT);
    if (shmid == -1) {
        perror("shmget");
        return(1);
    }
    shmptr = shmat(shmid, NULL, 0);
    if ((int)noff == -1) {
        perror("shmat");
        return(1);
    }

    noff = (off_t *)shmptr;
    *noff = 0;
    curdocid = (int *)(shmptr + sizeof(off_t));
    *curdocid = -1;
#else
    noff = 0;
    curdocid = -1;
#endif
    return(0);
}


/*
 * read a null terminated string in a data file
 * returns a pointer to a newly allocated string
 */
static char *indexer_readword(FILE **f) {
    char word[2 * WORDSZ + 1 + 1]; /* buffer for 2 words */
    char *newword;
    char c;
    int pos;


    if (*f == NULL) {
        return(strdup(""));
    }
    pos = 0;
    c = fgetc(*f);
    while (c != '\0' && !feof(*f)) {
        if (pos < 2 * WORDSZ + 1)
            word[pos++] = c;
        c = fgetc(*f);
    }
    if (pos == 2 * WORDSZ + 1) {
        printf("Warning: buffer size for indexer_readword() is too small\n");
    }
    if (feof(*f)) {
        fclose(*f);
        *f = NULL;
    }
    word[pos] = '\0';
    newword = strdup(word);
    return(newword);
}


/*
 * find lowest value
 */
static int indexer_datacompare(s_pos *val, int count) {
    int i, min;


    min = -1;
    for (i = 0; i < count; i++) {
        if (val[i].docid != -1) {
            min = i;
            break;
        }
    }
    for (i = 1; i < count; i++) {
        if (val[i].docid != -1
            && (val[i].docid < val[min].docid || (val[i].docid == val[min].docid && val[i].wordpos < val[min].wordpos)))
            min = i;
    }
    return(min);
}


static int indexer_wordmerge(FILE **infiles, char **allwords, int count, char *word, FILE *outfile, int incount, int *mapsize) {
    int i;
    int found[MAXFILES];
    FILE *f[MAXFILES];
    int matched = 0;
    int remaining;
    s_pos dataw[MAXFILES];
    int datacount[MAXFILES];
    int total;
    int lowest;
    int opos;
    int stopword;
    char *bitfield;
    int bitfieldsz;
    char *ptr;
    int istword;
    s_ff *ff;
    int ffc = 0, ffi = 0, ffpos, fftotal;
    off_t soff;


    /*
    * opened files may or may not contain informations for this word: we look for
    * concerned descriptors and store them in f[]
    */
    for (i = 0; i < count; i++) {
        if (strcmp(allwords[i], word) == 0) {
            f[matched++] = infiles[i];
            found[i] = 1;
        } else
            found[i] = 0;
    }

    /*
    * merge all positions informations into outfile
    */

    total = 0;
    for (i = 0; i < matched; i++) {
        fread(&datacount[i], sizeof(int), 1, f[i]);
        total += datacount[i];
    }
    
    /* read first values in source files */
    for (i = 0; i < matched; i++) {
        fread(&dataw[i], sizeof(s_pos), 1, f[i]);
        datacount[i]--;
    }
    
    /* check if the word is really two words following each other */
    ptr = word;
    istword = 0;
    while (*ptr != '\0') {
        if (*ptr == '\t') {
            istword = 1;
            break;
        }
        ptr++;
    }
    
    /*
     * a word is considered a "stop word" if the size required to save all
     * docid + positions is greater than saving a whole bitfield telling
     * whether the word is in the page or not
     * changed 2008-12-10: there are too many stop words for our 200 GB files
     * total > incount / (8 + 8) -> total > incount
     */
    remaining = matched;
    lowest = indexer_datacompare(dataw, matched);
    if (incount && total > incount && !istword) {
        *mapsize = bitfieldsz = (incount >> 3) + ((incount & 7) ? 1 : 0);
        bitfield = (char *)malloc(bitfieldsz);
        memset(bitfield, 0, bitfieldsz);
        bitfield[dataw[lowest].docid >> 3] ^= 1 << (dataw[lowest].docid & 7);
        stopword = 1;
        total = 1;
    } else {
        ffc = total / FFGAP;
        if (ffc) {
            ff = (s_ff *)malloc(sizeof(s_ff) * ffc);
            soff = ftello(outfile);
            fseek(outfile, sizeof(s_ff) * ffc, SEEK_CUR);
            fftotal = 1;
            ffpos = 0;
        }

        *mapsize = sizeof(s_pos);
        stopword = 0;
        fwrite(&dataw[lowest], sizeof(s_pos), 1, outfile);
    }
    opos = dataw[lowest].docid;
    while (remaining) {
        if (datacount[lowest] < 0) { /* simple sanity check */
            printf("datacount %d < 0 ? %d\n", lowest, datacount[lowest]);
            printf("pos %lld\n", ftello(f[lowest]));
            printf("word [%s]\n", word);
            exit(1);
        }

        if (datacount[lowest]) { /* read next value */
            fread(&dataw[lowest], sizeof(s_pos), 1, f[lowest]);
            datacount[lowest]--;
            lowest = indexer_datacompare(dataw, matched);
        } else { /* no more data to read in this file */
            dataw[lowest].docid = -1;
            remaining--;
            if (remaining)
                lowest = indexer_datacompare(dataw, matched);
            else
                continue;
        }
        if (stopword) {
            if (dataw[lowest].docid != opos) {
                bitfield[dataw[lowest].docid >> 3] ^= 1 << (dataw[lowest].docid & 7); /* set bit to 1 for this docid */
                total++;
                opos = dataw[lowest].docid;
            }
        } else {
             if (ffc) {
                 if ((fftotal % FFGAP) == 0) { /* write "skip/fast forward" information for this chunk */
                     ff[ffi].offset = *mapsize;
                     ff[ffi].docid = dataw[lowest].docid;
                     ff[ffi].pos = ffpos; /* we'll need to know where we start from once we have skipped the data */
                     ffi++;
                 }
                 fftotal++;
                 ffpos++;
             }
             fwrite(&dataw[lowest], sizeof(s_pos), 1, outfile);
             *mapsize += sizeof(s_pos);
             opos = dataw[lowest].docid;
         }
    }
    if (stopword) {
        fwrite(bitfield, 1, bitfieldsz, outfile);
        free(bitfield);
        total |= FLAG_BITFIELD;
    } else if (ffc) {
        fseeko(outfile, soff, SEEK_SET); /* go back to start of data for this word and write fast forward structures */
        fwrite(ff, sizeof(s_ff), ffc, outfile);
        free(ff);
        fseeko(outfile, 0, SEEK_END);
    }

    /*
    * skip to next word in source files
    */
    for (i = 0; i < count; i++) {
        if (found[i]) {
            /*
            * only read next word in files that we previously found to have *word
            */
            free(allwords[i]);
            allwords[i] = indexer_readword(&infiles[i]);
        }
    }
    return(total);
}


/*
 * compares given words and returns the lowest one
 */
static int indexer_findlowestword(char **word, unsigned int *hash, int count) {
    int i;
    long long diff;
    int lowest;


    lowest = 0;
    while (lowest < count && word[lowest][0] == '\0')
        lowest++;
    if (lowest == count) /* not found */
        return(-1);

    hash[lowest] = dhash_hash(word[lowest]);

    for (i = lowest + 1; i < count; i++) {
        if (word[i][0] != '\0') {
            hash[i] = dhash_hash(word[i]);
            diff = ((long long)(hash[lowest])) - ((long long)(hash[i]));
            if (diff > 0)
                lowest = i;
            else if (!diff) {
                if (strcmp(word[lowest], word[i]) > 0)
                    lowest = i;
            }
        }
    }
    return(lowest);
}


/*
 * merge source files
 */
static void indexer_mergefiles(char *outname, int incount) {
    char filename[100 + 1], **fnames = NULL;
    FILE *f[MAXFILES], *outf;
    int i, fc;
    DIR *d;
    struct dirent *dirent;
    char **fileword;
    unsigned int *filewordh;
    char idxword[2 * WORDSZ + 1 + 1]; /* 2 words + \t + \0 */
    int lowid;
    FILE *fdboff, *fdbdata;
    off_t offset;
    int total;
    char dbbuf[sizeof(off_t) + sizeof(int) + sizeof(int)];
    int mapsize;
    off_t dbdataoff;
    unsigned int h, oldh;
    int count;
    

    fc = 0;
    d = opendir(".");
    while (dirent = readdir(d)) {
        /*
        * open files matching idx.* in the current directory
        */
        if (!strncmp(dirent->d_name, "idx.", 4)) {
            if (fc == MAXFILES) {
                printf("too many files, increase MAXFILES\n");
                exit(1);
            }
            fnames = (char **)realloc(fnames, sizeof(char *) * (fc + 1));
            fnames[fc] = strdup(dirent->d_name); /* store filename for later deletion */
            if ((f[fc] = fopen(dirent->d_name, "r")) == NULL) {
                perror(dirent->d_name);
                exit(1);
            }
            fc++;
        }
    }
    closedir(d);

    snprintf(filename, 100, "%s.idx", outname);
    if ((outf = fopen(filename, "w")) == NULL) {
        perror(filename);
        exit(1);
    }
    
    snprintf(filename, 100, "%s.dboff", outname);
    if ((fdboff = fopen(filename, "w")) == NULL) {
        perror(filename);
        exit(1);
    }
    snprintf(filename, 100, "%s.dbdata", outname);
    if ((fdbdata = fopen(filename, "w")) == NULL) {
        perror(filename);
        exit(1);
    }

    fileword = (char **)malloc(sizeof(char *) * fc);
    filewordh = (unsigned int *)malloc(sizeof(unsigned int) * fc);
    for (i = 0; i < fc; i++) {
        fileword[i] = indexer_readword(&f[i]);
    }

    count = dbdataoff = oldh = 0;
	while ((lowid = indexer_findlowestword(fileword, filewordh, fc)) != -1) {
			/*
			* write offset to data in .idx file in .db with
			* word with lowest id as key
			*/
			offset = ftello(outf); /* current position in .idx file */
		strcpy(idxword, fileword[lowid]);
		total = indexer_wordmerge(f, fileword, fc, fileword[lowid], outf, incount, &mapsize);
		memcpy(dbbuf, &offset, 5); /* start of data (docids + positions) for this word in .idx file */
		memcpy(dbbuf + 5, &total, sizeof(int)); /* number of (docids + positions) couples */
		memcpy(dbbuf + 5 + sizeof(int), &mapsize, sizeof(int)); /* write exact data size for mmap() function */

		h = filewordh[lowid];
		if (h != oldh) {
	#ifdef PARANOID
			if (count > 255) {
				fprintf(stderr, "error: DHASHSZ should be (at least) > (number of different words) + 20%\n");
				fprintf(stderr, "merge continues, but some words *will* be missing\n");
				count = 255;
			}
	#endif
			db_append_dboff(fdboff, count, dbdataoff);
			dbdataoff = ftello(fdbdata);
			count = 0;
			oldh++;
			while (oldh < h) {
					/* fill idxoffset with empty buckets */
				db_append_dboff(fdboff, 0, 0);
				oldh++;
			}
		}
		db_append_dbdata(fdbdata, strlen(idxword), idxword, dbbuf);
		count++;
	}

	fclose(fdbdata);
	fclose(outf);

		/* write last dbdata offset in dboff */
	db_append_dboff(fdboff, count, dbdataoff);

		/* fill idxoffset to the end with empty buckets */
	h++;
	while (h < DHASHSZ) { /* fill idxoffset with empty buckets */
		db_append_dboff(fdboff, 0, 0);
		h++;
	}
	fclose(fdboff);

    /*
    * remove temporary idx.* files
    */
    for (i = 0; i < fc; i++) {
        unlink(fnames[i]);
        free(fnames[i]);
        free(fileword[i]);
    }
    free(fnames);
    free(fileword);
    free(filewordh);
}


static inline int indexer_wordcompare(const void *a, const void *b) {
    long long diff;


    diff = ((long long)(dhash_hash((*((s_word **)a))->word))) - ((long long)(dhash_hash((*((s_word **)b))->word)));
    if (!diff)
        return(strcmp((*((s_word **)a))->word, (*((s_word **)b))->word));

    return(diff);
}


static int indexer_wordtree_save(s_word *word, FILE *f) {
    fwrite(word->word, 1, strlen(word->word) + 1, f);
    fwrite(&word->count, sizeof(int), 1, f);
    fwrite(word->pos, sizeof(s_pos), word->count, f);
#ifdef DEBUG
    debugcount += word->count;
#endif

    free(word->word);
    free(word->pos);
    return(1);
}



static void indexer_wordtree_free(s_word *word) {
    free(word);
}


static void indexer_savetree(void) {
    FILE *fidx;
    char filename[100];
    static int count;
    s_word *word;
    unsigned int i;


    snprintf(filename, 100, "idx.%d%d%d", time(NULL), getpid(), count++);
    if ((fidx = fopen(filename, "w")) == NULL) {
        perror(filename);
        exit(1);
    }
    qsort(hash->array, hash->entries, sizeof(s_word *), indexer_wordcompare);
    for (i = 0; i < hash->entries; i++)
        indexer_wordtree_save(hash->array[i], fidx);

    hash_init(hash);
    idxcount = 0;
    fclose(fidx);
}


/* add (docid + position) to tree for the given string, only if this docid has not already been inserted before */
static void indexer_indexword_uniq(char *str, int docid, int wordpos) {
    s_word key, *wordptr;


    if (str[0] == '\0') {
        printf("string is empty\n"); /* shouldn't happen */
        return;
    }

    /* add word to hash */
    wordptr = hash_insert(hash, str, str + strlen(str));
    if (!wordptr)
        return; /* malloc failed */
    
    if (wordptr->count) {
        if (wordptr->pos[wordptr->count - 1].docid == docid)
            return;
    }
    wordptr->pos = (s_pos *)realloc(wordptr->pos, sizeof(s_pos) * (wordptr->count + 1));
    wordptr->pos[wordptr->count].docid = docid;
    wordptr->pos[wordptr->count].wordpos = wordpos;
    wordptr->count++;
    idxcount++;
}


/* same but keep all positions in a docid (used for substring lookup as we need to check that positions are following) */
static void indexer_indexword(char *str, int docid, int wordpos) {
    s_word key, *wordptr;


    if (str[0] == '\0') {
        printf("string is empty\n"); /* shouldn't happen */
        return;
    }

    /* add word to hash */
    wordptr = hash_insert(hash, str, str + strlen(str));
    if (!wordptr)
        return; /* malloc failed */
    
    wordptr->pos = (s_pos *)realloc(wordptr->pos, sizeof(s_pos) * (wordptr->count + 1));
    wordptr->pos[wordptr->count].docid = docid;
    wordptr->pos[wordptr->count].wordpos = wordpos;
    wordptr->count++;
    idxcount++;
}


static int indexer_readdata(FILE *f, indexer_read_func user_readdata, char **data, char **ndata, int *id) {
#ifndef NO_MULTI
    struct sembuf sb = {
        0, -1, 0
    };
    off_t off;


    /* lock */
    sb.sem_op = -1;
    semop(semid, &sb, 1);
    flock(fileno(f), LOCK_EX);
    off = *noff;
    fseeko(f, off, SEEK_SET);
#endif

    /* read URL */
    user_readdata(f, data, ndata, id);
    if (feof(f)) {
#ifndef NO_MULTI
        flock(fileno(f), LOCK_UN);
        sb.sem_op = 1;
        semop(semid, &sb, 1);
#endif
        return(0);
    }

    /* write current offset in data file for this idurl */
#ifndef NO_MULTI
    flock(fileno(fid), LOCK_EX);
    fseek(fid, 0, SEEK_END);
    fwrite(&off, sizeof(off_t), 1, fid);
    fflush(fid);
    flock(fileno(fid), LOCK_UN);

    /* unlock */
    *noff = ftello(f);
    flock(fileno(f), LOCK_UN);
    (*curdocid)++;
    *id = *curdocid; /* move docid to process space */
    sb.sem_op = 1;
    semop(semid, &sb, 1);
#else
    fwrite(&noff, sizeof(off_t), 1, fid);
    noff = ftello(f);
    curdocid++;
    *id = curdocid;
#endif
    return(1);
}


static void clearstring_latin(char *str) {
    char *ptr;
  

    ptr = str;
    while (*ptr != '\0') {
        *ptr = latinidx[(unsigned char)*ptr];
        ptr++;
    }
}


static void clearstring_latinlnk(char *str) {
    char *ptr;


    ptr = str;
    while (*ptr != '\0') {
        *ptr = latinidx_lnk[(unsigned char)*ptr];
        ptr++;
    }
}


static void indexer_loop(FILE *f, indexer_read_func user_readdata) {
    char *data, *ndata, *orig;
    char prevw[WORDSZ + 1], bufword[2 * WORDSZ + 1 + 1];
    char word[WORDSZ + 1];
    int ldone, l;
    int i;
    int wordsz, ws;
    char *ptr;
    int id;
    int wordpos, prevp;
    
    
    while (indexer_readdata(f, user_readdata, &data, &ndata, &id)) {
        wordpos = 0;
        
        if (data) {
            orig = strdup(data);
            clearstring_latin(data);
            clearstring_latinlnk(orig);
            prevw[0] = '\0';
            ldone = -1;
            i = 0;
            while (data[i] != '\0') {
                while (data[i] != '\0' && isspace(data[i])) {
                    i++;
                }
                if (data[i] != '\0') {
                    wordsz = 0;
                    ptr = data + i;
                    while (*ptr != '\0' && !isspace(*ptr)) {
                        ptr++;
                        wordsz++;
                    }
                    if (wordsz > WORDSZ)
                        wordsz = WORDSZ;
                    memcpy(word, data + i, wordsz);
                    word[wordsz] = '\0';

                    //indexer_indexword(word, id, wordpos);
                    indexer_indexword_uniq(word, id, wordpos);

                    if (prevw[0] != '\0' && wordpos != 16383) {
                        sprintf(bufword, "%s\t%s", prevw, word);
                        indexer_indexword(bufword, id, prevp);
                    }
                    strcpy(prevw, word);

                    /* if one or more special characters were found, index the word a second time without these characters */
                    if (i >= ldone) {
                        ws = 0;
                        l = 0;
                        ptr = orig + i;
                        while (*ptr != '\0' && !isspace(*ptr)) {
                            if (lnkchar[(unsigned char)*ptr])
                                l++;
                            ptr++;
                            ws++;
                        }
                        if (!l)
                            goto nolcar;
                        while (ws > 1 && endchar[(unsigned char)orig[i + ws - 1]]) {
                            ws--;
                            l--;
                        }
                        if (l && l != ws) { /* if only special characters, do not index */
                            if (ws > WORDSZ)
                                ws = WORDSZ;
                            memcpy(word, orig + i, ws);
                            word[ws] = '\0';

                            //indexer_indexword(word, id, wordpos);
                            indexer_indexword_uniq(word, id, wordpos);
                            ldone = i + ws;
                        }
                    }
            nolcar:
                    i += wordsz;
                }
                if (hash->entries > maxcount || idxcount > maxidx)
                    indexer_savetree();
                prevp = wordpos;
                if (wordpos != 16383)
                    wordpos++;
            }
            free(orig);
        }
        
        /* index the "no change" buffer */
        if (ndata) {
            ptr = ndata;
            while (*ptr != '\0') {
                wordsz = 0;
                while (wordsz < WORDSZ && *ptr != '\0' && *ptr != '\1') {
                    word[wordsz++] = *ptr;
                    ptr++;
                }
                word[wordsz] = '\0';
                indexer_indexword_uniq(word, id, wordpos);
                //indexer_indexword(word, id, wordpos);
                if (wordpos != 16383)
                    wordpos++;
                if (*ptr == '\1')
                    ptr++;
            }
        }
    }
    if (hash->entries) {
        indexer_savetree();
        hash_free(hash);
    }
}


int indexer(int forkc, char *pname, char *outname, char *inname, indexer_read_func user_readdata, int incount) {
    FILE *f;
    int ismain = 1;
    
    
    if (indexer_init(pname, outname))
        return(1);

    hash = hash_create(maxcount);
    if ((f = fopen(inname, "r")) == NULL) {
        perror(inname);
        return(1);
    }
    
#ifndef NO_MULTI
    while (forkc > 1) {
        if (fork() == 0) {
            ismain = 0;
            /* that's pretty much a bad idea to work on the same handler with 2 different processes */
            fclose(f);
            f = fopen(inname, "r");
            break;
        } else
            forkc--;
    }
#endif

    indexer_loop(f, user_readdata); /* this one should be fork()-able */

    fclose(f);
    fclose(fid);
#ifndef NO_MULTI
    if (!ismain)
        exit(0);

    while (wait(NULL) > 0)
        ;
    
    semctl(semid, 0, IPC_RMID);
    shmdt(shmptr);
    shmctl(shmid, IPC_RMID, NULL);
#endif

    indexer_mergefiles(outname, incount);
    return(0);
}
