/*--------------------------------------------------------------------------------
 * Copyright (c) 2008, Jean-Christophe Dufour. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 4. Neither the name of the copyright holder nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *--------------------------------------------------------------------------------
 * Title: hash
 * Revision: 0.1
 * Author: Jean-Christophe Dufour (jc@uhlu.com)
 * $Date: 2008/09/08 15:04:35 $
 * Description: hash library dedicated to libindexer
 *--------------------------------------------------------------------------------
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hash.h"



inline int hash_strcmp(char *p1, char *p2, char *end) {

    while (*p2 == *p1) {
        p2++;
        p1++;
        if (p2 == end)
            return (*p1);
    }
    return (1);
}



/*
 * Based on an algorithm by E. Knuth in The Art Of Computer Programming Volume 3
 */
inline unsigned int hash_hash (s_hash *hash, char *str, char *end) {
    register unsigned int h = (unsigned int)(end - str);

    while (str < end)
        h = ((h << 5) ^ (h >> 27)) ^ *str++;

    return(h % hash->sz);
}



void hash_init(s_hash *hash) {

    memset(hash->bucket, 0, hash->sz * sizeof(s_word *));
    hash->entries = 0;

    hash->cur_word = NULL;
    hash->cur_bucket = 0;
}

inline int hash_isprime(unsigned int n) {
    unsigned long long i;
  

    if (n < 2)
        return(0);

    for (i = 2; i * i < n + 1; i++)
        if (n % i == 0)
            return(0);
    
    return(1);
}
    
    
inline unsigned int hash_prime(unsigned int n) {
    
    while(!hash_isprime(n))
        n++;

    return(n);
}


s_hash *hash_create(unsigned int sz) {
    s_hash *hash;
    unsigned int i;


    hash = (s_hash *)malloc(sizeof(s_hash));
    if (hash == NULL)
        return(NULL);

    hash->sz = hash_prime((sz) ? sz : HASHSZ);
    hash->bucket = (s_word **)malloc(sizeof(s_word *) * hash->sz);
    if (hash->bucket == NULL)
        return(NULL);

    hash->array = (s_word **)malloc(sizeof(s_word *) * hash->sz);
    if (hash->array == NULL)
        return(NULL);

    hash_init(hash);

    return (hash);
}



s_word *hash_insert(s_hash *hash, char *word, char *end) {
    unsigned int h;
    s_word **w;
    int sz;



    h = hash_hash(hash, word, end);
    w = &(hash->bucket[h]);

    while (*w != NULL && (*w)->word != NULL && hash_strcmp((*w)->word, word, end))
        w = &((*w)->next);

    if (*w != NULL && (*w)->word != NULL)
        return (*w);

    *w = (s_word *)malloc(sizeof(s_word));
    if (*w == NULL)
        return (NULL);

    sz = end - word;
    (*w)->word = (char *)malloc(sizeof(char) * (sz + 1));
    if ((*w)->word == NULL)
        return (NULL);

    (*w)->next = NULL;
    memcpy((*w)->word, word, sz);
    (*w)->word[sz] = '\0';
    (*w)->pos = NULL;
    (*w)->count = 0;

    hash->array[hash->entries] = *w;
    hash->entries++;


    return (*w);
}   

    
void hash_free(s_hash *hash) {

    free(hash->bucket);
    free(hash->array);
    free(hash);
}
